| package |
package := Package name: 'CU Ghoul'.
package paxVersion: 1;
	basicComment: 'Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org

Tool for presenting the contents of Dolphin''s post-mortem files, i.e. the ones produced by a crash dump, or by an unhandled exception in a deployed application  It presents a debugger-like view of the stack traces.

The terms under which you may use this software are:
	You must not claim that you wrote it.
	You must not claim that you own it.
	You use it at your own risk.

I''d appreciate it if you didn''t distribute modified versions without at least trying to tell me about the changes you thought worthwhile.

	-- chris


History:

4.00
-	Now use PackageResourceLocator for icons, etc.

3.00
-	Now picks up source pane settings from SmalltalkWorkspace.
-	Now gives option to remove files from recent files list if it can''t open them.
-	Now has (partially implemented) generic browsing commands.
-	Better behaviour when double-clicking over undefined method.
-	Can now drag method/class name from frames list to a workspace.
-	Now tries to be a bit clever about the initially selected frame in the backtrace.
-	Can now read data from the Windows clipboard (it''s a ''recent file'')
-	Now accepts files draged from Windows if ''DH Shell Data Transfer'' is installed (doesn''t work across image save).
-	Parsing, although still simplistic, is significantly less fragile.
-	Added toggle between hex and decimal parsing of IP numbers (to override rhe default assumptions).
-	Added config option to force the date format it uses for parsing timestamps.
-	Added config option to control how the background color is faded.

2.00
-	First release.'.

package basicPackageVersion: '4.007'.

package basicScriptAt: #postinstall put: '(Package manager packageNamed: ''CU Ghoul'')
	propertyAt: #ExternalResourceFileNames
	put: #(
		''Resources\Ghoul.ico''
	).
!!
'.

package classNames
	add: #GhoulDummyFrame;
	add: #GhoulModel;
	add: #GhoulModelAbstract;
	add: #GhoulShell;
	add: #GhoulStackFrame;
	add: #GhoulStackFramePresenter;
	add: #GhoulStackTrace;
	add: #GhoulStackTracePresenter;
	add: #VirtualGhoulModel;
	yourself.

package binaryGlobalNames: (Set new
	yourself).

package globalAliases: (Set new
	yourself).

package setPrerequisites: (IdentitySet new
	add: 'CU Tools Base';
	add: '..\Object Arts\Dolphin\IDE\Base\Development System';
	add: '..\Object Arts\Dolphin\Base\Dolphin';
	add: '..\Object Arts\Dolphin\MVP\Views\Common Controls\Dolphin Common Controls';
	add: '..\Object Arts\Dolphin\MVP\Dialogs\Common\Dolphin Common Dialogs';
	add: '..\Object Arts\Dolphin\MVP\Models\List\Dolphin List Models';
	add: '..\Object Arts\Dolphin\MVP\Presenters\List\Dolphin List Presenter';
	add: '..\Object Arts\Dolphin\MVP\Base\Dolphin MVP Base';
	add: '..\Object Arts\Dolphin\MVP\Views\Scintilla\Dolphin Scintilla View';
	add: '..\Object Arts\Dolphin\MVP\Presenters\Text\Dolphin Text Presenter';
	add: '..\Object Arts\Dolphin\MVP\Type Converters\Dolphin Type Converters';
	yourself).

package!

"Class Definitions"!

Object subclass: #GhoulDummyFrame
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #GhoulModelAbstract
	instanceVariableNames: 'stackTraces'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #GhoulStackFrame
	instanceVariableNames: 'trace ipString methodName receiver arguments locals selector receiverClassName receiverClassIsMeta methodClassName methodClassIsMeta'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #GhoulStackTrace
	instanceVariableNames: 'errorMessage stackFrames isCrashdump timestamp ipStringBase'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: 'dummy'!
GhoulModelAbstract subclass: #GhoulModel
	instanceVariableNames: 'internalTimestamp filename'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
GhoulModelAbstract subclass: #VirtualGhoulModel
	instanceVariableNames: 'name source'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Presenter subclass: #GhoulStackFramePresenter
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Presenter subclass: #GhoulStackTracePresenter
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
CUToolShell subclass: #GhoulShell
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: 'recentFiles'!

"Global Aliases"!


"Loose Methods"!

"End of package definition"!

"Source Globals"!

"Classes"!

GhoulDummyFrame guid: (GUID fromString: '{3E9DB246-EF60-42E7-A5ED-C67427F44B7C}')!
GhoulDummyFrame comment: 'Copyright  Chris Uppal, 2004.
chris.uppal@metagnostic.org

Simple imitation of a GhoulStackFrame that is used to mark the end of stack traces that have been truncated artificially.'!
!GhoulDummyFrame categoriesForClass!Unclassified! !
!GhoulDummyFrame methodsFor!

displayOn: aStream

	aStream nextPutAll: '<... truncated ...>'.!

isDummy

	^ true.!

method

	^ nil.!

methodClass

	^ nil.!

methodIsCurrent

	^ false.!

methodIsDefined

	^ false.!

methodTimestamp

	^ nil.!

receiverClass

	nil.! !
!GhoulDummyFrame categoriesFor: #displayOn:!displaying!public! !
!GhoulDummyFrame categoriesFor: #isDummy!public!testing! !
!GhoulDummyFrame categoriesFor: #method!accessing!public! !
!GhoulDummyFrame categoriesFor: #methodClass!accessing!public! !
!GhoulDummyFrame categoriesFor: #methodIsCurrent!public!testing! !
!GhoulDummyFrame categoriesFor: #methodIsDefined!public!testing! !
!GhoulDummyFrame categoriesFor: #methodTimestamp!accessing!public! !
!GhoulDummyFrame categoriesFor: #receiverClass!accessing!public! !

GhoulModelAbstract guid: (GUID fromString: '{54A9A509-8130-45D6-AFEC-62B4FF370C97}')!
GhoulModelAbstract comment: 'Copyright  Chris Uppal, 2004.
chris.uppal@metagnostic.org
'!
!GhoulModelAbstract categoriesForClass!Unclassified! !
!GhoulModelAbstract methodsFor!

canRefresh
	"answer whether we are able re-read out defintion from our data source"

	self subclassResponsibility.!

externalFileHasChanged
	"answer true if the post-moretem file seems to have changed since we
	last scanned it"

	"since we don't have one..."
	^ false.!

initialize
	"private -- establish a coherent initial state"

	stackTraces := OrderedCollection new.
!

isVirtual
	"answer whether we are a virtual instance (i.e. not representing a physiscal file)"

	"the default"
	^ false.!

printOn: aStream
	"append a programmer-centric representation of this record to aStrream"

	aStream
		basicPrint: self;
		space;
		display: self.!

refresh
	"re-read out defintion from our data source"

	self withDataSourceDo: [:stream | stackTraces := GhoulStackTrace readAllFromStream: stream].!

stackTraces
	"answer a list of our stack traces"

	^ stackTraces.
!

withDataSourceDo: a1Block
	"private -- answer the result of evaluating a1Block, passing in a ReadStream
	reading from our data source"

	self subclassResponsibility.! !
!GhoulModelAbstract categoriesFor: #canRefresh!public!testing! !
!GhoulModelAbstract categoriesFor: #externalFileHasChanged!public!testing!timestamping! !
!GhoulModelAbstract categoriesFor: #initialize!initializing!private! !
!GhoulModelAbstract categoriesFor: #isVirtual!public!testing! !
!GhoulModelAbstract categoriesFor: #printOn:!printing!public! !
!GhoulModelAbstract categoriesFor: #refresh!public!reading! !
!GhoulModelAbstract categoriesFor: #stackTraces!accessing!public! !
!GhoulModelAbstract categoriesFor: #withDataSourceDo:!private!reading! !

!GhoulModelAbstract class methodsFor!

new
	"private -- answer a new instance with default initialisation"

	^ (self basicNew)
		initialize;
		yourself.! !
!GhoulModelAbstract class categoriesFor: #new!instance creation!private! !

GhoulStackFrame guid: (GUID fromString: '{E730FFE9-2796-4ABE-90BF-A1C79F7BCCAF}')!
GhoulStackFrame comment: 'Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org
'!
!GhoulStackFrame categoriesForClass!Unclassified! !
!GhoulStackFrame methodsFor!

arguments
	"answer our Array of arguments"

	^ arguments.
!

argumentsAndLocals
	"answer an Array of all our arguments and locals (does not include the reciever)"

	^ arguments , locals.
!

displayOn: aStream
	"append a user-centric description of ourself to aStream"

	aStream display: methodName.!

hasVariables
	"answer whether we've been given a list of variables at all (sometimes
	that data's not included in the dump)"

	^ receiver notNil.!

initialize
	"private -- establish a coherent (if iffy) intiial state"

	ipString := ''.
	methodName := ''.
	receiver := nil.
	arguments := OrderedCollection new.
	locals := OrderedCollection new.
!

ip
	"answer our ip number, decoding the string according to our owning stack trance's
	idea of the format"

	| radix ip |

	radix := trace ipStringBase.
	ip := 0.

	ipString do:
		[:ch || digit |
		digit := ch asUppercase digitValue.
		(digit between: 0 and: radix - 1) ifFalse:  [^ ip].
		ip := ip * radix + digit].

	^ ip.
!

isBlank
	"answer whether we've been given any data at all"

	#CUtodo.  "this /shouldn't/ be needed"

	^ methodName isEmpty.!

isDummy
	"answer whether we are a dummy stack frame"

	^ false.!

locals
	"answer our Array locals"

	^ locals.
!

method
	"answer the actual method in the current image that corresponds
	to this frame's method name.  Or nil in the likely event that there
	isn't one"

	^ self methodClass ifNotNil: [:it | it compiledMethodAt: selector ifAbsent: [nil]].!

methodClass
	"answer the class contianing our method or nil if that class is not defined in this image"

	^ Smalltalk at: methodClassName ifPresent: [:it | methodClassIsMeta ifTrue: [it class] ifFalse: [it]].!

methodClassIsMeta
	"answer whether our method class is meta"

	^ methodClassIsMeta.
!

methodClassName
	"answer the Symbol name of out method class"

	^ methodClassName.
!

methodIsCurrent
	"answer whether the method we correspond to is present in this image
	and has not been re-defined"

	| then |

	then := trace timestamp ifNil: [^ true "can't tell"].

	^ self methodIsDefined and: [self methodTimestamp
						ifNil: [true "can't tell"]
						ifNotNil: [:current | current <= then]].!

methodIsDefined
	"answer whether the method we correspond to is defined in this image"

	^ self method notNil.!

methodName
	"answer our method name (the full form that appears in the backtrace)"

	^ methodName.
!

methodTimestamp
	"if we can work out a timestamp for the current definition of this
	method, then answer it, otherwise answer nil.
	This implementation uses STS"

	| info edition |

	(self respondsTo: #sourceControl) ifFalse: [^ nil].

	info := self method ifNotNil: [:it | self sourceControl getVersionInfoFor: it].
	edition := info ifNotNil: [:it | it edition].

	^ edition ifNotNil: [:it | it timestamp].
!

parseMethodName
	"private -- if possible, then parse out the method and class names from our full #methodName"

	| string index |

	"very, very, grubby..."
	string := methodName.
	(string beginsWith: '[] in') ifTrue: [string := string allButFirst: 6].

	#CUtodo.  "it would be nice to handle 'doIt' style names, and (unbound) ones, though they shouldn't
			be generated by deployed exes (I think)"

	string := string trimBlanks.
	index := string indexOfSubCollection: '>>'.
	index = 0 ifTrue: [^ nil].

	"gosh, we've found the method name already"
	selector := (string copyFrom: index + 2) asSymbol.
	string := string first: index - 1.

	"the remainder is the name of the class, or in the form Subclass(Superclass)"
	index := string indexOf: $(.
	index = 0
		ifTrue:
			[receiverClassName := string.
			methodClassName := string]
		ifFalse:
			[receiverClassName := string first: index - 1.
			methodClassName := string copyFrom: index + 1 to: string size - 1].

	"now check for metas"
	(receiverClassIsMeta := receiverClassName endsWith: ' class') ifTrue: [receiverClassName := receiverClassName allButLast: 6].
	(methodClassIsMeta := methodClassName endsWith: ' class') ifTrue: [methodClassName := methodClassName allButLast: 6].

	"and finally convert to Symbols"
	receiverClassName := receiverClassName asSymbol.
	methodClassName := methodClassName asSymbol.
!

printOn: aStream
	"append a programmer-centric representation of this record to aStrream"

	aStream
		basicPrint: self;
		space;
		display: self.!

readDefinitionFrom: aReadStream
	"private -- populate ourself by reading the data from aStream"

	"the data should look something like:

		06FA04FC: cf 06FA04E9, sp 06FA050C, bp 06ED5320, ip 23, ZipFileEntry>>contentsAsBinary:

	we discard everything except the IP and the remainder of the line (which, note, may contain commas)"

	"skip up to and over ', ip'"
	(aReadStream skipToAll: ', ip') ifFalse:
		[^ self].

	"read the IP"
	ipString := aReadStream
			skipSeparators;
			upTo: $,.

	"and the method name is everything that's left"
	methodName := aReadStream
				skipSeparators;
				upToEnd.

	"but now that needs to be broken up too..."
	self parseMethodName.!

readElementFrom: aReadStream 
	"private -- read ONE OF our reciever, args, etc, from the given stream"

	| name value |
	aReadStream atEnd ifTrue: [^self].
	name := aReadStream nextWord.
	value := aReadStream upToEnd.
	name = 'receiver:' 
		ifTrue: [receiver := value]
		ifFalse: 
			[(name beginsWith: 'arg') 
				ifTrue: [arguments addLast: value]
				ifFalse: 
					[locals addLast: ((value beginsWith: 'temp') 
								
								ifTrue: [value copyFrom: (value indexOf: $:) + 2] ifFalse: [value])]]!

readElementsFrom: aReadStream
	"private -- read our reciever, args, etc, from the given stream"

	"NB: under some circumstances that I don't understand, Dolphin will
	omit all the locals and and separating blank line.  Also this way of expressing
	the parse does not depend on blank lines to detect the end of the elements"
	[aReadStream atEnd] whileFalse:
	[| line |
		('{<' includes: aReadStream peek) ifTrue: [^ self].
		line := aReadStream nextLine.
		self readElementFrom: line readStream].!

receiver
	"answer our receiver (the String name)"

	^ receiver ifNil: [''].
!

receiverClass
	"answer the class of our receiver or nil if that class is not defined in this image"

	^ Smalltalk at: receiverClassName ifPresent: [:it | receiverClassIsMeta ifTrue: [it class] ifFalse: [it]].!

receiverClassIsMeta
	"answer whether our receiver class is meta"

	^ receiverClassIsMeta.
!

receiverClassName
	"answer the Symbol name of out receiver class"

	^ receiverClassName.
!

selector
	"answer the Symbol selector of our method"

	^ selector.
!

trace: aStackTrace
	"private -- set the stack trace of which we are an element"

	trace := aStackTrace.! !
!GhoulStackFrame categoriesFor: #arguments!accessing!public! !
!GhoulStackFrame categoriesFor: #argumentsAndLocals!accessing!public! !
!GhoulStackFrame categoriesFor: #displayOn:!displaying!public! !
!GhoulStackFrame categoriesFor: #hasVariables!public!testing! !
!GhoulStackFrame categoriesFor: #initialize!initializing!private! !
!GhoulStackFrame categoriesFor: #ip!accessing!public! !
!GhoulStackFrame categoriesFor: #isBlank!public!testing! !
!GhoulStackFrame categoriesFor: #isDummy!public!testing! !
!GhoulStackFrame categoriesFor: #locals!accessing!public! !
!GhoulStackFrame categoriesFor: #method!accessing!public! !
!GhoulStackFrame categoriesFor: #methodClass!accessing!public! !
!GhoulStackFrame categoriesFor: #methodClassIsMeta!accessing!public!testing! !
!GhoulStackFrame categoriesFor: #methodClassName!accessing!public! !
!GhoulStackFrame categoriesFor: #methodIsCurrent!public!testing! !
!GhoulStackFrame categoriesFor: #methodIsDefined!public!testing! !
!GhoulStackFrame categoriesFor: #methodName!accessing!public! !
!GhoulStackFrame categoriesFor: #methodTimestamp!accessing!public! !
!GhoulStackFrame categoriesFor: #parseMethodName!private!reading! !
!GhoulStackFrame categoriesFor: #printOn:!printing!public! !
!GhoulStackFrame categoriesFor: #readDefinitionFrom:!private!reading! !
!GhoulStackFrame categoriesFor: #readElementFrom:!private!reading! !
!GhoulStackFrame categoriesFor: #readElementsFrom:!private!reading! !
!GhoulStackFrame categoriesFor: #receiver!accessing!public! !
!GhoulStackFrame categoriesFor: #receiverClass!accessing!public! !
!GhoulStackFrame categoriesFor: #receiverClassIsMeta!accessing!public!testing! !
!GhoulStackFrame categoriesFor: #receiverClassName!accessing!public! !
!GhoulStackFrame categoriesFor: #selector!accessing!public! !
!GhoulStackFrame categoriesFor: #trace:!initializing!private! !

!GhoulStackFrame class methodsFor!

definedBy: aString in: aStackTrace
	"answer a new instance with basic data (not arguments or temporaries) defined by the given
	string"

	^ (self new)
		trace: aStackTrace;
		readDefinitionFrom: aString readStream;
		yourself.!

new
	"private -- should only be created by reading a post-mortem file"

	^ (self basicNew)
		initialize;
		yourself.! !
!GhoulStackFrame class categoriesFor: #definedBy:in:!instance creation!public! !
!GhoulStackFrame class categoriesFor: #new!instance creation!private! !

GhoulStackTrace guid: (GUID fromString: '{42E2B3E3-705B-4ED2-971D-E083D9D26DFA}')!
GhoulStackTrace comment: 'Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org

NB: The parsing here is *extremely* quick-and-dirty.  Just a succession of hacks, in fact.  I''d put some effort in to cleaning it up, except that it doesn''t seem worthwhile if -- as I suspect -- the format of these files will change in the next version (D6) of Dolphin.'!
!GhoulStackTrace categoriesForClass!Unclassified! !
!GhoulStackTrace methodsFor!

assumedDateFormat
	"answer the date format to assume when we are parsing files.
	NB: usually nil (which means we use the default for the current locale)"

	^ GhoulShell dumpFileDateFormat.!

crashdumpEndMarker
	"answer the string that marks the end of one trace in a crashdump file"

	^ '***** End of crash report *****'.!

displayOn: aStream
	"append a user-centric description of ourself to aStream"

	aStream display: errorMessage.!

errorMessage
	"answer our error message"

	^ errorMessage.
!

firstSignificantFrame
	"answer the frame that looks most likely to be interesting, i.e. not part of the
	error handling mechanism itself"

	| boring interesting |

	#CUtodo. "review this list"
	boring := (Set new)
			add: (Error compiledMethodAt: #defaultAction);
			add: (Exception compiledMethodAt: #signal);
			add: (Exception class compiledMethodAt: #signal);
			add: (Exception class compiledMethodAt: #signal:);
			add: (Exception class compiledMethodAt: #signal:with:);
			add: (Exception class compiledMethodAt: #signalWith:);
			add: (SessionManager compiledMethodAt: #logError:);
			add: (SessionManager compiledMethodAt: #unhandledException:);
			add: (SessionManager compiledMethodAt: #onUnhandledError:);
			add: (VMLibrary compiledMethodAt: #dump:path:stackDepth:walkbackDepth:);
			add: (VMLibrary compiledMethodAt: #crashDump:);
			add: (MessageNotUnderstood class compiledMethodAt: #receiver:message:);
			yourself.

	"we want the first method that is not boring *and* hasn't been called via a boring one"
	stackFrames asArray do:
		[:each |
		((boring includes: each method) "or: ['*error*' match: each methodName]")
			ifTrue: [interesting := nil]
			ifFalse: [interesting isNil ifTrue: [interesting := each]]].

	"may be nil"
	^ interesting.
		!

initialize
	"private -- establish a coherent initial state"

	errorMessage := ''.
	stackFrames := OrderedCollection new.!

ipStringBase
	"answer the base (10 or 16) to use when decoding IP numbers.  Initialy
	this is set from the kind of trace we represent (crash:16 dump:10) but can
	be overriden"

	^ ipStringBase ifNil: [isCrashdump ifTrue: [16] ifFalse: [10]].!

ipStringBase: anIntegerOrNil
	"set the base to use when decoding IP numbers"

	^ ipStringBase := anIntegerOrNil.!

isCrashdump
	"answer whether we are in crashdump format, as created by actual crashes or
	by VMLibrary>>crashDump:, as opposed to the normal dump format created by
	VMLibrary>>dump:path:stackDepth:walkbackDepth:"

	^ isCrashdump.!

parseOutTimestamp
	"private -- attempt to read the timestamp from our error message.
	Answers a TimeStamp or nil"

	| stream time date |

	stream := errorMessage readStream.

	[time := Time readFrom: stream.
	stream skip: 1; skipSeparators.
	"apparently, under WinXP Dolphin writes the date in locale-specific
	format, but under Win2K it is always in US format.  Sigh..."
	date := Date readFrom: stream format: self assumedDateFormat]
		on: InvalidFormat
		do: [:err | ^ nil].

	^ TimeStamp date: date time: time.!

printOn: aStream
	"append a programmer-centric representation of this record to aStrream"

	aStream
		basicPrint: self;
		space;
		display: self.!

readErrorMessageFrom: aReadStream
	"private -- read the timestamp and error message the given stream.
	NB: we consume the input up to and including the next 'marker'"

	| line |

	[aReadStream atEnd ifTrue: ["sneaky way to raise the correct exception" aReadStream next].
	line := aReadStream nextLine.
	(line beginsWith: '*--') ifTrue:
		[errorMessage := errorMessage trimBlanks.
		timestamp := self parseOutTimestamp.
		^ self].
	errorMessage := errorMessage , ' ' , line]
		repeat.!

readFrom: aReadStream
	"private -- populate ourself by reading the next complete stack trace from the given stream"

	self
		readHeaderFrom: aReadStream;
		readStackFramesFrom: aReadStream;
		readTrailerFrom: aReadStream.!

readHeaderFrom: aReadStream
	"private -- read the timestamp and error message the given stream.
	NB: we consume the input up to and including the 'Stack Back Trace' marker"

	self
		readPrologFrom: aReadStream;
		readErrorMessageFrom: aReadStream;
		skipToStackTraceFrom: aReadStream.
!

readPrologFrom: aReadStream
	"private -- scan the stream looking for the start of the next trace or crashdump record.
	Will throw EOF exception if the stream does not contain anything interesting"

	| line end |

	[aReadStream atEnd ifTrue: ["sneaky way to raise the correct exception" aReadStream next].
	line := aReadStream nextLine.
	(line indexOfSubCollection: 'Dolphin Virtual Machine Dump Report') > 0 ifTrue: [isCrashdump := false. ^ self].
	(line indexOfSubCollection: 'Dolphin Crash Dump Report') > 0 ifTrue: [isCrashdump := true. ^ self]]
		repeat.
!

readStackFramesFrom: aReadStream 
	"private -- read the stack frames from the given stream.
	NB: we consume the input up to and including the 'Bottom of stack' marker"

	| line end truncated |
	end := self stackEndMarker.
	truncated := self stackTruncatedMarker.
	
	[| frame |
	aReadStream atEnd 
		ifTrue: 
			[stackFrames add: GhoulDummyFrame new.
			^self].
	(aReadStream peekFor: ${) 
		ifTrue: 
			[line := aReadStream upTo: $}.
			aReadStream skipSeparators]
		ifFalse: [line := aReadStream nextLine].
	line = end ifTrue: [^self].
	line = truncated 
		ifTrue: 
			[stackFrames add: GhoulDummyFrame new.
			^self].
		
	frame := GhoulStackFrame definedBy: line in: self.
	frame readElementsFrom: aReadStream.
	frame isBlank ifFalse: [stackFrames add: frame]] 
			repeat!

readTrailerFrom: aReadStream
	"private -- read data from aReadStream up to the end of the trace"

	| line end |

	end := isCrashdump
		ifTrue: [self crashdumpEndMarker]
		ifFalse: [self traceEndMarker].

	"the #atEnd test shouldn't be needed, just for added robustness"
	[aReadStream atEnd or: [(line := aReadStream nextLine) = end]] whileFalse.
!

skipToStackTraceFrom: aReadStream
	"private -- advance up aReadStream untill we find the begining of the next stack trace"

	| line end |

	end := self stackStartMarker.

	[aReadStream atEnd ifTrue: ["sneaky way to raise the correct exception" aReadStream next].
	line := aReadStream nextLine.
	line = end] whileFalse.
!

stackEndMarker
	"answer the string that marks the end of the 'Stack Back Trace' in a most-mortem file"

	^ '<Bottom of stack>'.!

stackFrames
	"answer our list of stack frames"

	^ stackFrames.
!

stackStartMarker
	"answer the string that marks the begining of the 'Stack Back Trace' in a post-mortem file"

	^ '*----> Stack Back Trace <----*'.!

stackTruncatedMarker
	"answer the string that marks the end of a truncated 'Stack Back Trace' in a most-mortem file"

	^ '<...more...>'.!

timestamp
	"answer the TimeStamp deduced from our error message, or nil"

	^ timestamp.!

traceEndMarker
	"answer the string that marks the end of one trace in a most-mortem file"

	^ '***** End of dump *****'.! !
!GhoulStackTrace categoriesFor: #assumedDateFormat!constants!public! !
!GhoulStackTrace categoriesFor: #crashdumpEndMarker!constants!public! !
!GhoulStackTrace categoriesFor: #displayOn:!displaying!public! !
!GhoulStackTrace categoriesFor: #errorMessage!accessing!public! !
!GhoulStackTrace categoriesFor: #firstSignificantFrame!accessing!public! !
!GhoulStackTrace categoriesFor: #initialize!initializing!private! !
!GhoulStackTrace categoriesFor: #ipStringBase!accessing!public! !
!GhoulStackTrace categoriesFor: #ipStringBase:!accessing!public! !
!GhoulStackTrace categoriesFor: #isCrashdump!public!testing! !
!GhoulStackTrace categoriesFor: #parseOutTimestamp!private!reading! !
!GhoulStackTrace categoriesFor: #printOn:!printing!public! !
!GhoulStackTrace categoriesFor: #readErrorMessageFrom:!private!reading! !
!GhoulStackTrace categoriesFor: #readFrom:!initializing!private!reading! !
!GhoulStackTrace categoriesFor: #readHeaderFrom:!private!reading! !
!GhoulStackTrace categoriesFor: #readPrologFrom:!private!reading! !
!GhoulStackTrace categoriesFor: #readStackFramesFrom:!private!reading! !
!GhoulStackTrace categoriesFor: #readTrailerFrom:!private!reading! !
!GhoulStackTrace categoriesFor: #skipToStackTraceFrom:!private!reading! !
!GhoulStackTrace categoriesFor: #stackEndMarker!constants!public! !
!GhoulStackTrace categoriesFor: #stackFrames!accessing!public! !
!GhoulStackTrace categoriesFor: #stackStartMarker!constants!public! !
!GhoulStackTrace categoriesFor: #stackTruncatedMarker!constants!public! !
!GhoulStackTrace categoriesFor: #timestamp!accessing!public! !
!GhoulStackTrace categoriesFor: #traceEndMarker!constants!public! !

!GhoulStackTrace class methodsFor!

dummy
	"answer a dummy instance"

	dummy isNil ifTrue: [dummy := self new].

	^ dummy.!

new
	"private -- should only be created by reading a post-mortem file"

	^ (self basicNew)
		initialize;
		yourself.!

readAllFromFile: aFilename
	"answer a list of instances created by scanning the named file"

	| stream |

	stream := FileStream read: aFilename.
	[^ self readAllFromStream: stream]
		ensure: [stream close].

!

readAllFromStream: aReadStream
	"answer a list of instances created by scanning the given stream"

	| all |

	all := OrderedCollection new.

	#CUtodo.  "we should not be supressing errors like this!!"
	[aReadStream skipSeparators; atEnd] whileFalse:
		[[all add: (self readFromStream: aReadStream)]
			on: Exception
			do: [:err | ^ all]].

	^ all.
!

readFromStream: aReadStream
	"answer a new instances created by scanning the next trace from given stream"

	^ (self new)
		readFrom: aReadStream;
		yourself.!

uninitialize
	"private -- class-side tear-down.

		self uninitialize.
	"

	dummy := nil.
! !
!GhoulStackTrace class categoriesFor: #dummy!instance creation!public! !
!GhoulStackTrace class categoriesFor: #new!instance creation!private! !
!GhoulStackTrace class categoriesFor: #readAllFromFile:!public!reading! !
!GhoulStackTrace class categoriesFor: #readAllFromStream:!public!reading! !
!GhoulStackTrace class categoriesFor: #readFromStream:!instance creation!public!reading! !
!GhoulStackTrace class categoriesFor: #uninitialize!initialization!private! !

GhoulModel guid: (GUID fromString: '{AA0C57BD-227E-4BDD-B1BB-D25DAC6B69AA}')!
GhoulModel comment: 'Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org

One of these represents the contents of a Dolphin post-mortem file.'!
!GhoulModel categoriesForClass!Unclassified! !
!GhoulModel methodsFor!

canRefresh
	"answer whether we are able re-read out defintion from our data source"

	^ filename notNil.!

displayOn: aStream
	"append a user-centric description of ourself to aStream"

	aStream display: filename.!

externalFileHasChanged
	"answer true if the post-moretem file seems to have changed since we
	last scanned it"

	| ext int |

	ext := self externalTimestamp.
	int := self internalTimestamp.

	"If there's no external time, then assume it's OK"
	ext isNil ifTrue: [^false].

	"If there's no internal time, but there is an external one, then
	it's probably not OK"
	int isNil ifTrue: [^true].

	^ext asInteger > int asInteger.!

externalTimestamp
	"answer the timestamp (FILETIME) associated with
	the real post-moretem file or nil if there isn't one."

	filename isNil ifTrue: [^ nil].
	^ [File lastWriteTime: filename]
		on: Error
		do: [:e | ^ nil].!

filename
	"answer the name of the file that we represent"

	^ filename.!

filename: aFilename
	"private -- set the name of the file that we represent"

	filename := aFilename.
	self resetInternalTimestamp.
!

initialize
	"private -- establish a coherent initial state"

	super initialize.

	filename := '<none>'.
	internalTimestamp := FILETIME now.
!

internalTimestamp
	"answer the timestamp (FILETIME) associated with the
	receiver's internal representation."

	^ internalTimestamp.!

isVirtual
	"answer whether we are a virtual instance (i.e. not representing a physiscal file)"

	^ false.!

readFrom: aFilename
	"private -- replace our data from the named file"

	self
		filename: aFilename;
		refresh.
!

refresh
	"re-read out defintion from our file"

	super refresh.
	self resetInternalTimestamp.
!

resetInternalTimestamp
	"force the receiver's internal timestamp to be that
	of its external representation (and so possibly nil)."

	internalTimestamp := self externalTimestamp.!

withDataSourceDo: a1Block
	"private -- answer the result of evaluating a1Block, passing in a ReadStream
	reading from our data source"

	| stream |

	stream := FileStream read: filename.
	^ [a1Block value: stream] ensure: [stream close].
! !
!GhoulModel categoriesFor: #canRefresh!public!testing! !
!GhoulModel categoriesFor: #displayOn:!displaying!public! !
!GhoulModel categoriesFor: #externalFileHasChanged!public!testing!timestamping! !
!GhoulModel categoriesFor: #externalTimestamp!public!timestamping! !
!GhoulModel categoriesFor: #filename!accessing!public! !
!GhoulModel categoriesFor: #filename:!initializing!private! !
!GhoulModel categoriesFor: #initialize!initializing!private! !
!GhoulModel categoriesFor: #internalTimestamp!accessing!public!timestamping! !
!GhoulModel categoriesFor: #isVirtual!public!testing! !
!GhoulModel categoriesFor: #readFrom:!initializing!private!reading! !
!GhoulModel categoriesFor: #refresh!public!reading! !
!GhoulModel categoriesFor: #resetInternalTimestamp!public!timestamping! !
!GhoulModel categoriesFor: #withDataSourceDo:!private!reading! !

!GhoulModel class methodsFor!

fileTypes
	"answer an Array of file types that are normally used for post-mortem files"

	^ #(
		('Dolphin post-mortem files (*.errors, *.dump)' '*.errors;*.dump')
		('All Files (*.*)' '*.*')
	).!

fromFile: aFilename
	"answer a new instance read from the named file"

	^ (self new)
		readFrom: aFilename;
		yourself.! !
!GhoulModel class categoriesFor: #fileTypes!constants!public! !
!GhoulModel class categoriesFor: #fromFile:!instance creation!public! !

VirtualGhoulModel guid: (GUID fromString: '{02984579-7460-4071-B7A5-956828BF25FD}')!
VirtualGhoulModel comment: 'Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org

These are similar to GhoulModels except that they aren''t read from files.'!
!VirtualGhoulModel categoriesForClass!Unclassified! !
!VirtualGhoulModel methodsFor!

canRefresh
	"answer whether we are able re-read out defintion from our data source"

	^ source notNil.!

displayOn: aStream
	"append a user-centric desciption of this object to aStream"

	aStream display: name.!

initialize
	"private -- establish a coherent initial state"

	super initialize.

	name := '<none>'.
	source := nil.
!

isVirtual
	"answer whether we are a virtual instance (i.e. not representing a physiscal file)"

	^ true.!

name
	"answer the name of the data source that we represent"

	^ name.!

name: aString
	"private -- set 	the name of the data source that we represent"

	name := aString.!

source: a0Block
	"private -- set 	the <nildadicValuable> that supplies us with data"

	source := a0Block.!

withDataSourceDo: a1Block
	"private -- answer the result of evaluating a1Block, passing in a ReadStream
	reading from our data source"

	^ a1Block value: source value.! !
!VirtualGhoulModel categoriesFor: #canRefresh!public!testing! !
!VirtualGhoulModel categoriesFor: #displayOn:!displaying!public! !
!VirtualGhoulModel categoriesFor: #initialize!initializing!private! !
!VirtualGhoulModel categoriesFor: #isVirtual!public!testing! !
!VirtualGhoulModel categoriesFor: #name!accessing!public! !
!VirtualGhoulModel categoriesFor: #name:!initializing!private! !
!VirtualGhoulModel categoriesFor: #source:!initializing!private! !
!VirtualGhoulModel categoriesFor: #withDataSourceDo:!private!reading! !

!VirtualGhoulModel class methodsFor!

dummy
	"answer a dummy instance"

	^ self new.!

fromClipboard
	"answer a new instance read from the clipboard"

	^ self
		name: '<Clipboard>'
		source: [Clipboard current getText readStream].!

name: aString source: a0Block
	"answer a new instance that gets data by evaluating a0block (which is
	expected to answer a ReadStream)"

	^ (self new)
		name: aString;
		source: a0Block;
		refresh;
		yourself.! !
!VirtualGhoulModel class categoriesFor: #dummy!instance creation!public! !
!VirtualGhoulModel class categoriesFor: #fromClipboard!instance creation!public! !
!VirtualGhoulModel class categoriesFor: #name:source:!instance creation!public! !

GhoulStackFramePresenter guid: (GUID fromString: '{C1B9B404-0863-46EB-89BE-0177E061A971}')!
GhoulStackFramePresenter comment: 'Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org
'!
!GhoulStackFramePresenter categoriesForClass!Unclassified! !
!GhoulStackFramePresenter methodsFor!

createComponents
	"private -- create presenters in order that they may be bound into MVP triads"

	self
		add: (TextPresenter new) name: 'MethodSource';
		add: (ListPresenter new) name: 'Variables';
		yourself.

	^ super createComponents.
!

fadeFactor
	"private -- answer how much to fade the source pane's
	background colour by"

	^ GhoulShell sourcePaneFadeFactor.!

findIpInMap: aDebugMap 
	"private -- lookup our current IP in the given 'map', which is a SortedCollection
	of associations mapping starts of IP ranges to <whatever>"

	"the stored IP is of the *next* instruction"

	| ip |
	ip := self model ip - 1.

	"in case we weren't able to parse the IP properly"
	ip < 0 ifTrue: [^nil].
	aDebugMap do: [:each | (each key includes: ip) ifTrue: [^each value]].



	"probably shouldn't get here..."
	^nil!

findIpInTextMap: aDebugMap 
	"private -- lookup our current IP in the given 'map', which is a SortedCollection
	of associations mapping starts of IP ranges to <whatever>"

	"the stored IP is of the *next* instruction"

	| ip lastKey lastValue |
	
	ip := self model ip - 1.

	"in case we weren't able to parse the IP properly"
	ip < 0 ifTrue: [^nil].

	"there seems to be a bug in the maps such that a sensible first
	element is often missing.  This hacks around the problem"
	ip := ip max: aDebugMap first key.

	"we are looking for the last element with a key that incliudes the IP"
	aDebugMap reverseDo: [:each | each key <= ip ifTrue: [^each value]].

	"probably shouldn't get here..."
	^nil!

importWorkspaceSettings
	"private -- copy across the default settings from SmalltakWorkspace to
	our source text presenter (which doesn't pick them up by default because
	it's not a workspace)"

	| source |

	source := Smalltalk at: #SmalltalkWorkspace ifAbsent: [^ self].
	(self methodSourcePresenter view)
		backcolor: (source defaultBackcolor fadedBy: self fadeFactor);
		font: source actualFont;
		wordWrap: source wordWrap.!

methodSourcePresenter
	"private -- answer the presenter named 'MethodSource'"

	^ self presenterNamed: 'MethodSource'.
!

model: aGhoulStackFrame
	"set the model that we will display"

	"deliberately don't supersend, to avoid the blasted flicker"
	"super model: aGhoulStackFrame.	"
	model := aGhoulStackFrame.

	(aGhoulStackFrame isNil or: [aGhoulStackFrame isDummy]) ifTrue:
		[self updateNone.
		^ self].

	"change this to use #methodIsCurrent if a more pessimistic behaviour is desired"
	aGhoulStackFrame methodIsDefined
		ifTrue: [self updateFull]
		ifFalse: [self updateSimple].!

onSettingsChanged
	"private -- the global settings have changed, update accordingly"

	"unfortunately, this is only called for changes to GhoulShell settings, not SmalltalkWorkspace"
	self importWorkspaceSettings.!

onViewOpened
	"private -- called when our view is fully open"

	self importWorkspaceSettings.

	super onViewOpened.!

queryCommand: aCommandQuery
	"set the enabledness of the command represented by aCommandQuery"

	| cmd |

	super queryCommand: aCommandQuery.

	cmd := aCommandQuery commandSymbol.

	#CUtodo.	"quick hack to prevent these commands getting passed to the stack frame"
	(#( #browseIt #browseDefinitions  #browseReferences ) includes: cmd) ifTrue:
		[aCommandQuery isEnabled: false; receiver: self].
!

updateFull
	"update our display from our model, using the full debug info
	from the method"

	| method info |

	method := self model method.

	info := method getDebugInfo .

	self
		updateSourceWith: info;
		updateVariablesWith: info.!

updateNone
	"update our display to show nothing"

	self
		updateSourceNone;
		updateVariablesNone.
!

updateSimple
	"update our display from our model, not attempting to use the information
	from this image"

	self
		updateSourceSimple;
		updateVariablesSimple.
!

updateSourceNone
	"private -- update ourself to show neither source nor an warning message"

	self methodSourcePresenter view isEnabled: false.

	self methodSourcePresenter model value: nil.
!

updateSourceSimple
	"private -- update our source from our model, not attempting to use the information
	from this image (and therefore not, in fact, displaying any source ;-)"

	| stream class |

	"grey-out the source pane"
	self methodSourcePresenter view isEnabled: false.

	self model isDummy ifTrue:
		[self methodSourcePresenter model value: nil.
		^ self].

	stream := String writeStream.

	stream nextPutAll: self model methodClassName.

	class := self model methodClass.
	class isNil ifFalse:
		[class isMeta ifTrue: [stream nextPutAll: ' class'].
		stream nextPutAll: '>>'.
		stream nextPutAll: self model selector].

	stream nextPutAll: ' is not defined in this image'.

	self methodSourcePresenter model value: stream contents.
!

updateSourceWith: aDebugInfo 
	"private -- update our source from our model, using the given debug information
	from this image"

	| range |
	
	self methodSourcePresenter model value: self model method getSource.
	range := self findIpInTextMap: aDebugInfo textMap.
	range isNil ifFalse: [self methodSourcePresenter selectionRange: range].

	"grey-out the source pane if the source is stale"
	self methodSourcePresenter view isEnabled: self model methodIsCurrent!

updateVariablesNone
	"private -- update ourself to show no variables"

	self variablesPresenter view isEnabled: false.
	self variablesPresenter list: #().
!

updateVariablesSimple
	"private -- update our variables from our model, not attempting to use the information
	from this image"

	| variables method |

	self model hasVariables ifFalse: [^ self updateVariablesNone].

	variables := OrderedCollection new.

	variables add: ('self' -> self model receiver).

	self model arguments keysAndValuesDo: [:i :each | variables add: (('arg %d' sprintfWith: i) -> each)].
	self model locals keysAndValuesDo: [:i :each | variables add: (('temp %d' sprintfWith: i) -> each)].

	self variablesPresenter list: variables.
	self variablesPresenter view isEnabled: true.!

updateVariablesWith: aDebugInfo 
	"private -- update our variables from our model, using the given debug information
	from this image"

	| variables locals i |
	
	self model hasVariables ifFalse: [^self updateVariablesNone].
	locals := self findIpInMap: aDebugInfo tempsMap.

	"just in case"
	locals isNil ifTrue: [locals := #()].

	"bug fix -- ultimately caused becase the raw DebgInfo has locals like:
		'aMouseEvent item  '
	which cases #subStrings: to add an unwanted item.
	The temp maps are, in any case, pretty flaky"
	[locals notEmpty and: [locals last isEmpty]] whileTrue: [locals := locals allButLast].
	variables := OrderedCollection new.
	variables add: 'self' -> self model receiver.
	i := 1.
	self model argumentsAndLocals do: 
			[:each | 
			| name |
			name := locals at: i ifAbsent: [Array with: ('_stack %d' sprintfWith: i - locals size)].
			variables add: name first -> each.
			i := i + 1].
	self variablesPresenter list: variables.
	self variablesPresenter view isEnabled: true!

variablesPresenter
	"private -- answer the presenter named 'Variables'"

	^ self presenterNamed: 'Variables'.
! !
!GhoulStackFramePresenter categoriesFor: #createComponents!initializing!private!subpresenters! !
!GhoulStackFramePresenter categoriesFor: #fadeFactor!helpers!private! !
!GhoulStackFramePresenter categoriesFor: #findIpInMap:!helpers!private! !
!GhoulStackFramePresenter categoriesFor: #findIpInTextMap:!helpers!private! !
!GhoulStackFramePresenter categoriesFor: #importWorkspaceSettings!helpers!private! !
!GhoulStackFramePresenter categoriesFor: #methodSourcePresenter!private!subpresenters! !
!GhoulStackFramePresenter categoriesFor: #model:!initializing!models!public!updating! !
!GhoulStackFramePresenter categoriesFor: #onSettingsChanged!event handling!private! !
!GhoulStackFramePresenter categoriesFor: #onViewOpened!event handling!private! !
!GhoulStackFramePresenter categoriesFor: #queryCommand:!commands!menus!public! !
!GhoulStackFramePresenter categoriesFor: #updateFull!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateNone!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateSimple!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateSourceNone!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateSourceSimple!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateSourceWith:!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateVariablesNone!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateVariablesSimple!private!updating! !
!GhoulStackFramePresenter categoriesFor: #updateVariablesWith:!private!updating! !
!GhoulStackFramePresenter categoriesFor: #variablesPresenter!private!subpresenters! !

!GhoulStackFramePresenter class methodsFor!

onSettingsChanged
	"private -- the global settings have changed, update our instances accordingly"

	self allSubinstances do: [:each | each onSettingsChanged].!

resource_Default_view
	"Answer the literal data from which the 'Default view' resource can be reconstituted.
	DO NOT EDIT OR RECATEGORIZE THIS METHOD.

	If you wish to modify this resource evaluate:
	ViewComposer openOn: (ResourceIdentifier class: self selector: #resource_Default_view)
	"

	^#(#'!!STL' 3 788558 10 ##(Smalltalk.STBViewProxy)  8 ##(Smalltalk.ContainerView)  98 15 0 0 98 2 8 1409286144 131073 416 0 524550 ##(Smalltalk.ColorRef)  8 4278190080 0 5 0 0 0 416 1180166 ##(Smalltalk.ProportionalLayout)  234 240 98 0 16 234 256 98 4 410 8 ##(Smalltalk.ScintillaView)  98 46 0 416 98 2 8 1445007684 1 640 0 498 8 4278190080 0 21 265030 4 ##(Smalltalk.Menu)  0 16 98 17 984134 2 ##(Smalltalk.CommandMenuItem)  1 1180998 4 ##(Smalltalk.CommandDescription)  8 #accept 8 '&Accept' 1 1 0 0 0 802 1 834 8 #reformatAccept 8 'Ref&ormat/Accept' 1 1 0 0 0 983366 1 ##(Smalltalk.DividerMenuItem)  4097 754 0 16 98 0 8 'Auto-&correct' 8 #autoCorrectMenu 134217729 0 0 0 0 0 754 0 16 98 16 802 1 834 8 #renameNode 8 'Re&name <1d>...' 1 1 0 0 0 962 4097 802 1 834 8 #extractToTemporary 8 'Extract to &Temporary...' 1 1 0 0 0 802 1 834 8 #extractMethod 8 'E&xtract Method...' 1 1 0 0 0 802 1 834 8 #extractToComponent 8 'Extract to &Component...' 1 5 0 0 0 962 4097 802 1 834 8 #inlineMessage 8 'Inline &Message' 1 1 0 0 0 754 0 16 98 0 8 'Impl&ement Message In' 8 #implementMessageMenu 134217729 0 0 0 0 0 962 4097 802 1 834 8 #inlineTemporary 8 '&Inline Temporary' 1 1 0 0 0 802 1 834 8 #moveTempToInnerScope 8 'Move to Inner &Scope' 1 1 0 0 0 802 1 834 8 #convertTempToInstVar 8 'Con&vert to Instance Variable' 1 1 0 0 0 962 4097 802 1 834 8 #addParameterToMessage 8 '&Add Parameter to <1d>...' 1 1 0 0 0 802 1 834 8 #inlineParameter 8 'In&line Parameter' 1 1 0 0 0 802 1 834 8 #removeParameter 8 'Remove &Parameter' 1 1 0 0 0 8 'Refactorin&gs' 8 #codeRefactoringsMenu 1 263494 3 ##(Smalltalk.Icon)  0 16 1114638 ##(Smalltalk.STBSingletonProxy)  8 ##(Smalltalk.ImageRelativeFileLocator)  8 #current 8 'Refactoring.ico' 2032142 ##(Smalltalk.STBExternalResourceLibraryProxy)  8 'dolphindr006.dll' 0 0 0 0 0 754 0 16 98 2 802 1 834 8 #reformatSource 8 '&Source' 1 1 0 0 0 802 1 834 8 #reformatComment 8 '&Comment' 1 1 0 0 0 8 'Re&format' 8 #reformatMenu 1 0 0 0 0 0 962 4097 754 0 16 98 8 802 1 834 8 #undo 8 '&Undo' 1 1 1954 0 16 2000 8 'EditUndo.ico' 2080 0 0 962 4097 802 1 834 8 #cutSelection 8 'Cu&t' 1 1 1954 0 16 2000 8 'EditCut.ico' 2080 0 0 802 1 834 8 #copySelection 8 '&Copy' 1 1 1954 0 16 2000 8 'EditCopy.ico' 2080 0 0 802 1 834 8 #pasteClipboard 8 '&Paste' 1 1 1954 0 16 2000 8 'EditPaste.ico' 2080 0 0 802 1 834 8 #clearSelection 8 'De&lete' 1 1 1954 0 16 2000 8 'EditClear.ico' 2080 0 0 962 4097 802 1 834 8 #selectAll 8 '&Select All' 1 1 0 0 0 8 '&Edit' 0 134217729 0 0 0 0 0 754 0 16 98 10 802 1 834 8 #inspectWorkspacePool 8 'Variab&les' 1 1 0 0 0 802 1 834 8 #cloneNew 8 'Clo&ne' 1 1 1954 0 16 2000 8 'SmalltalkWorkspace.ico' 2080 0 0 962 4097 802 1 834 8 #toggleAutoCompletion 8 'Auto-complete' 1 1 0 0 0 802 1 834 8 #toggleIndentationGuides 8 'Indentation &Guides' 1 1 0 0 0 802 1 834 8 #toggleLineNumbers 8 'Line N&umbers' 1 1 0 0 0 802 1 834 8 #toggleLineEndings 8 'Line &Endings' 1 1 0 0 0 802 1 834 8 #toggleStyling 8 '&Syntax Coloring' 1 1 0 0 0 802 1 834 8 #toggleWhitespace 8 'W&hitespace' 1 1 0 0 0 802 1 834 8 #toggleWordWrap 8 '&Word Wrap' 1 1 0 0 0 8 'Wor&kspace' 0 134217729 0 0 0 0 0 962 4097 802 1 834 8 #browseIt 8 'Bro&wse It' 1 1 1954 0 16 2000 8 'ClassBrowserShell.ico' 2080 0 0 802 1 834 8 #displayIt 8 '&Display It' 1 1 1954 0 16 2000 8 'DisplayIt.ico' 2080 0 0 802 1 834 8 #evaluateIt 8 'E&valuate It' 1 1 1954 0 16 2000 8 'EvaluateIt.ico' 2080 0 0 802 1 834 8 #inspectIt 8 '&Inspect It' 1 1 1954 0 16 2000 8 'InspectIt.ico' 2080 0 0 802 1 834 8 #debugIt 8 'Deb&ug It' 1 1 1954 0 16 2000 8 'Debugger.ico' 2080 0 0 962 4097 754 0 16 98 4 802 2097153 834 8 #browseDefinitions 8 'Defi&nitions of <1d>' 1 1 0 0 0 802 1 834 8 #browseReferences 8 '&References to <1d>' 1 1 0 0 0 962 4097 802 1 834 8 #browseMessage 8 '<1d>' 1 1 0 0 0 8 '&Browse' 0 1 0 0 0 0 0 8 '&Workspace' 0 134217729 0 0 0 0 0 0 0 640 0 482564749 852486 ##(Smalltalk.NullConverter)  0 0 11 0 234 256 98 56 8 #unarySelector 1182726 ##(Smalltalk.ScintillaTextStyle)  25 786694 ##(Smalltalk.IndexedColor)  33554465 0 3 0 0 0 0 4464 8 'Unary selectors (method signature)' 0 0 8 #comment 4482 39 4514 33554437 0 9 0 0 0 0 4560 8 'Comments in method source' 0 0 8 #literalCharacter 4482 11 4514 33554441 0 1 0 0 0 0 4624 8 'Literal character constants, e.g. $A' 0 0 8 #tempOpenBar 4482 13 0 0 9 0 0 0 0 4688 8 'Temporary declarations opening bar' 0 0 8 #keywordSelector 4482 21 4528 0 3 0 0 0 0 4736 8 'Components of multi-keyword message selectors' 0 0 8 #argDecl 4482 43 0 0 11 0 0 0 0 4784 8 'Argument declaration in method signature' 0 0 8 #keywordMessage 4482 45 4528 0 1 0 0 0 0 4832 4768 0 0 8 #literalSymbol 4482 5 4656 0 1 0 0 0 0 4864 8 'Literal symbol constants, e.g. #abc' 0 0 8 #literalBytes 4482 19 4514 33554439 0 1 0 0 0 0 4912 8 'Literal byte arrays, e.g. #[0 1 2]' 0 0 8 #binarySelector 4482 37 4528 0 3 0 0 0 0 4976 8 'Binary in-fix selectors such as + and - (method signature)' 0 0 8 #literalArray 4482 9 0 0 3 0 0 0 0 5024 8 'Opening/closing token of literal array, i.e. #()' 0 0 8 #unaryMessage 4482 15 4528 0 1 0 0 0 0 5072 8 'Unary (no argument) messages' 0 0 8 #identifier 4482 7 0 0 1 0 0 0 0 5120 8 'Variables references (instance, temporary and class)' 0 0 8 #indentGuide 4482 75 4514 33554447 0 1 0 0 0 0 5168 8 'Indentation guides, when visible' 0 0 8 #braceHighlight 4482 69 4528 0 3 0 0 0 0 5232 8 'Matching brace, when brace highlighting enabled' 0 0 8 #normal 4482 1 0 0 1 0 0 0 0 5280 8 'Default text style' 0 0 8 #tempDecl 4482 3 0 0 9 0 0 0 0 5328 8 'Temporary variable declaration' 0 0 8 #literalString 4482 23 4514 33554443 0 1 0 0 0 0 5376 8 'Literal string constants, e.g. ''abc''' 0 0 8 #binaryMessage 4482 33 4528 0 1 0 0 0 0 5440 8 'Binary in-fix messages such as + and -' 0 0 8 #illegal 4482 31 4514 33554459 0 3 0 0 0 0 5488 8 'Illegal characters, e.g. ' 0 0 8 #tag 4482 49 4514 33554457 0 9 0 0 0 0 5552 8 'Primitive or external call tag, e.g. <primitive: 1>' 0 0 8 #literalNumber 4482 27 4514 33554435 0 1 0 0 0 0 5616 8 'Numeric literal constants, e.g. 1.2e6' 0 0 8 #tempCloseBar 4482 41 0 0 9 0 0 0 0 5680 8 'Temporary declarations closing bar' 0 0 8 #blockArgDecl 4482 17 0 0 9 0 0 0 0 5728 8 'Block argument declaration' 0 0 8 #specialCharacter 4482 29 0 0 1 0 0 0 0 5776 8 'Special characters, e.g. normal and block parentheses' 0 0 8 #braceMismatch 4482 71 5520 0 3 0 0 0 0 5824 8 'Mismatched brace, when brace highlighting enabled' 0 0 8 #literalPseudo 4482 35 4514 33554445 0 1 0 0 0 0 5872 8 'The literal constants true, false and nil' 0 0 8 #assignment 4482 47 0 0 9 0 0 0 0 5936 8 'Assignment operation, i.e. :=' 0 0 98 40 5296 5344 4880 5136 5040 4640 4704 5088 5744 4928 4752 5392 4496 5632 5792 5504 5456 5888 4992 4576 5696 4800 4848 5952 5568 0 0 0 0 0 0 0 0 0 5248 5840 0 5184 0 0 1377542 ##(Smalltalk.SmalltalkMethodStyler)  1 0 0 32 202 208 592 234 256 98 2 8 #default 1639942 ##(Smalltalk.ScintillaMarkerDefinition)  1 1 4514 33554433 4514 33554471 640 8 #circle 202 208 592 0 63 0 0 0 0 0 5200 0 0 0 0 0 234 256 98 6 5024 8 '()' 5776 8 '()[]<>' 4912 8 '[]' 8 '' 3 234 256 98 2 8 #container 4432 0 0 0 0 3 0 0 983302 ##(Smalltalk.MessageSequence)  202 208 98 11 721670 ##(Smalltalk.MessageSend)  8 #createAt:extent: 98 2 328198 ##(Smalltalk.Point)  1 451 6466 751 441 640 6402 8 #contextMenu: 98 1 768 640 6402 8 #selectionRange: 98 1 525062 ##(Smalltalk.Interval)  3 1 3 640 6402 8 #isTextModified: 98 1 32 640 6402 8 #modificationEventMask: 98 1 9215 640 6402 8 #hoverTime: 98 1 1001 640 6402 8 #indicatorDefinitions: 98 1 98 2 1836038 ##(Smalltalk.ScintillaIndicatorDefinition)  1 640 33554459 3 6850 3 640 33554465 3 640 6402 8 #wordWrap: 98 1 16 640 6402 8 #margins: 98 1 98 3 984582 ##(Smalltalk.ScintillaMargin)  1 640 1 3 32 1 7010 3 640 1 1 32 67108863 7010 5 640 1 1 32 1 640 6402 8 #backspaceUnindents: 98 1 16 640 6402 8 #tabIndents: 98 1 16 640 983302 ##(Smalltalk.WINDOWPLACEMENT)  8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 0 0 0 0 225 0 0 0 119 1 0 0 189 1 0 0] 98 0 6466 193 193 0 27 8 'MethodSource' 410 8 ##(Smalltalk.ListView)  98 30 0 416 98 2 8 1140887629 1025 7264 590662 2 ##(Smalltalk.ListModel)  202 208 592 0 1994 8 ##(Smalltalk.SearchPolicy)  8 #identity 498 528 0 5 0 0 0 7264 0 8 4294902139 8 ##(Smalltalk.BasicListAbstract)  0 1994 8 ##(Smalltalk.IconImageManager)  2032 0 0 0 0 0 0 202 208 98 2 920646 5 ##(Smalltalk.ListViewColumn)  8 'Variable' 401 8 #left 7472 8 ##(Smalltalk.SortedCollection)  459270 ##(Smalltalk.Message)  8 #key 98 0 0 7264 0 1 0 0 7554 8 'Value' 343 7600 7472 7616 7634 8 #value 7680 0 7264 0 3 0 0 8 #report 592 0 131171 0 0 6338 202 208 98 2 6402 6432 98 2 6466 1 1 6466 751 441 7264 6402 8 #text: 98 1 8 'Variable' 7264 7170 8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 0 0 0 0 0 0 0 0 119 1 0 0 220 0 0 0] 98 0 7232 0 27 8 'Variables' 0 6338 202 208 98 1 6402 6432 98 2 6466 3839 21 6466 751 891 416 7170 8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 127 7 0 0 10 0 0 0 246 8 0 0 199 1 0 0] 98 3 7264 410 8 ##(Smalltalk.Splitter)  98 12 0 416 98 2 8 1140850688 1 8176 0 498 528 0 517 0 0 0 8176 6338 202 208 98 1 6402 6432 98 2 6466 1 441 6466 751 11 8176 7170 8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 0 0 0 0 220 0 0 0 119 1 0 0 225 0 0 0] 98 0 7232 0 27 640 7232 0 27 )! !
!GhoulStackFramePresenter class categoriesFor: #onSettingsChanged!event handling!private! !
!GhoulStackFramePresenter class categoriesFor: #resource_Default_view!public!resources-views! !

GhoulStackTracePresenter guid: (GUID fromString: '{DF8DF29B-2C45-43EE-8A58-1CCC07BE7D46}')!
GhoulStackTracePresenter comment: 'Copyright  Chris Uppal, 2004.
chris.uppal@metagnostic.org
'!
!GhoulStackTracePresenter categoriesForClass!Unclassified! !
!GhoulStackTracePresenter methodsFor!

browseDefinitions
	"command -- the generic F12 command"

	| selector |

	self canBrowseReferences ifFalse: [^ self].

	selector := self selectedFrame selector ifNil: [^ self].

	SmalltalkSystem current browseDefinitionsOf: selector.!

browseIt
	"command -- the generic ctrl+B command"

	| frame |

	frame := self selectedFrame ifNil: [^ self].

	"try to browse the corresponding method"
	frame method ifNotNil: [:it | ^ it browse].

	"or, failing that, the class"
	frame methodClass ifNotNil: [:it | ^ it browse].

	"give up..."
	MessageBox
		warning: ('Class ' , frame methodClassName , ' is not defined in this image')
		caption: self topShell class toolName.
!

browseReferences
	"command -- the generic shift+F12 command"

	| selector |

	self canBrowseReferences ifFalse: [^ self].

	selector := self selectedFrame selector ifNil: [^ self].

	SmalltalkSystem current browseReferencesTo: selector.!

canBrowseDefinitions
	"private -- can we issue the generic F12 command ?"

	^ self selectedFrame notNil.!

canBrowseIt
	"private -- can we issue the generic ctrl+B command ?"

	^ self selectedFrame notNil.!

canBrowseReferences
	"private -- can we issue the generic shift+F12 command ?"

	^ self selectedFrame notNil.!

createComponents
	"private -- create presenters in order that they may be bound into MVP triads"

	self
		add: (ListPresenter new) name: 'FrameList';
		add: (GhoulStackFramePresenter new) name: 'StackFrame';
		yourself.

	^ super createComponents.
!

createSchematicWiring
	"private -- arrange triggering between our components"

	self frameListPresenter
		when: #drag: send: #onDragFromFrames: to: self;
		when: #selectionChanged send: #onFrameSelected to: self;
		when: #actionPerformed send: #onFrameActioned to: self.

	^ super createSchematicWiring.
!

frameListPresenter
	"private -- answer the presenter named 'FrameList'"

	^ self presenterNamed: 'FrameList'.
!

model: aGhoulStackTrace
	"set the model that we will display"

	| frames |

	"deliberately don't supersend, to avoid the blasted flicker"
	"super model: aGhoulStackTrace.	"
	model := aGhoulStackTrace.

	frames := aGhoulStackTrace stackFrames.

	self frameListPresenter list: frames.

	self frameListPresenter selectionOrNil: (aGhoulStackTrace firstSignificantFrame).!

onDragFromFrames: aSession
	"private -- a d&d session is attempting to drag from the frames list.
	Update the session accordingly"

	aSession dragObjects do:
		[:each || object |
			object :=  each object.
			each format: #String data: object displayString].

!

onFrameActioned
	"private -- the user has double =clicked (or similar) an entry in the stack trace.
	Attempt to browse the current definition of that method"

	self browseIt.!

onFrameSelected
	"private -- a stack frame has been selected, update the stack frame presenter
	accordingly"

	self stackFramePresenter model: self selectedFrame.!

queryCommand: aCommandQuery
	"set the enabledness of the command represented by aCommandQuery"

	| cmd enabled |

	super queryCommand: aCommandQuery.

	cmd := aCommandQuery commandSymbol.
	enabled := aCommandQuery isEnabled.

	cmd == #browseIt ifTrue: [enabled := self canBrowseIt].
	cmd == #browseDefinitions ifTrue: [enabled := self canBrowseDefinitions].
	cmd == #browseReferences ifTrue: [enabled := self canBrowseReferences].

	aCommandQuery isEnabled: enabled.
!

selectedFrame
	"private -- answer the currently selected frame, if there is one,
	or nil, if not"

	^ self frameListPresenter selectionOrNil.!

showMaps

	| method |

	method := self selectedFrame ifNotNil: [:frame | frame method].
	method isNil ifTrue: [^ self].
	method inspect.
	Smalltalk at: #TablePresenter ifPresent:
		[:it |
		(it showOn: (method tempsMap collect: [:each | Array with: each key with: each value]))
			topShell caption: ('Temps: ' , method displayString).
		(it showOn: (method textMap collect: [:each | Array with: each key with: each value]))
			topShell caption: ('Text: ' , method displayString).
		].
!

stackFramePresenter
	"private -- answer the presenter named 'StackFrame'"

	^ self presenterNamed: 'StackFrame'.
! !
!GhoulStackTracePresenter categoriesFor: #browseDefinitions!commands!public! !
!GhoulStackTracePresenter categoriesFor: #browseIt!commands!public! !
!GhoulStackTracePresenter categoriesFor: #browseReferences!commands!public! !
!GhoulStackTracePresenter categoriesFor: #canBrowseDefinitions!commands!private! !
!GhoulStackTracePresenter categoriesFor: #canBrowseIt!commands!private! !
!GhoulStackTracePresenter categoriesFor: #canBrowseReferences!commands!private! !
!GhoulStackTracePresenter categoriesFor: #createComponents!initializing!private!subpresenters! !
!GhoulStackTracePresenter categoriesFor: #createSchematicWiring!event handling!initializing!private!subpresenters! !
!GhoulStackTracePresenter categoriesFor: #frameListPresenter!private!subpresenters! !
!GhoulStackTracePresenter categoriesFor: #model:!initializing!public! !
!GhoulStackTracePresenter categoriesFor: #onDragFromFrames:!event handling!private! !
!GhoulStackTracePresenter categoriesFor: #onFrameActioned!event handling!private! !
!GhoulStackTracePresenter categoriesFor: #onFrameSelected!event handling!private! !
!GhoulStackTracePresenter categoriesFor: #queryCommand:!commands!menus!public! !
!GhoulStackTracePresenter categoriesFor: #selectedFrame!accessing!private! !
!GhoulStackTracePresenter categoriesFor: #showMaps!helpers!private! !
!GhoulStackTracePresenter categoriesFor: #stackFramePresenter!private!subpresenters! !

!GhoulStackTracePresenter class methodsFor!

defaultModel

	^ GhoulStackTrace dummy.!

resource_Default_view
	"Answer the literal data from which the 'Default view' resource can be reconstituted.
	DO NOT EDIT OR RECATEGORIZE THIS METHOD.

	If you wish to modify this resource evaluate:
	ViewComposer openOn: (ResourceIdentifier class: self selector: #resource_Default_view)
	"

	^#(#'!!STL' 3 788558 10 ##(Smalltalk.STBViewProxy)  8 ##(Smalltalk.ContainerView)  98 15 0 0 98 2 8 1140850688 131073 416 0 524550 ##(Smalltalk.ColorRef)  8 4278190080 0 5 0 0 0 416 1180166 ##(Smalltalk.ProportionalLayout)  234 240 98 2 410 8 ##(Smalltalk.ReferenceView)  98 14 0 416 98 2 8 1140850688 131073 608 0 0 0 5 0 0 0 608 1180166 ##(Smalltalk.ResourceIdentifier)  8 ##(Smalltalk.GhoulStackFramePresenter)  8 #resource_Default_view 0 983302 ##(Smalltalk.MessageSequence)  202 208 98 1 721670 ##(Smalltalk.MessageSend)  8 #createAt:extent: 98 2 328198 ##(Smalltalk.Point)  317 1 882 625 831 608 983302 ##(Smalltalk.WINDOWPLACEMENT)  8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 158 0 0 0 0 0 0 0 214 1 0 0 159 1 0 0] 98 0 882 193 193 0 27 5 32 234 256 98 4 410 8 ##(Smalltalk.ListView)  98 30 0 416 98 2 8 1140936781 1025 1040 590662 2 ##(Smalltalk.ListModel)  202 208 976 0 1114638 ##(Smalltalk.STBSingletonProxy)  8 ##(Smalltalk.SearchPolicy)  8 #identity 498 528 0 15 265030 4 ##(Smalltalk.Menu)  0 16 98 3 984134 2 ##(Smalltalk.CommandMenuItem)  1 1180998 4 ##(Smalltalk.CommandDescription)  8 #browseIt 8 '&Browse It' 1 1 0 0 0 1298 1 1330 8 #browseDefinitions 8 'Browse &definitions' 1 1 0 0 0 1298 1 1330 8 #browseReferences 8 'Browse &references' 1 1 0 0 0 8 '' 0 1 0 0 0 0 0 0 0 1040 0 8 4294901963 8 ##(Smalltalk.BasicListAbstract)  0 1178 8 ##(Smalltalk.IconImageManager)  8 #current 0 0 0 0 0 0 202 208 98 1 920646 5 ##(Smalltalk.ListViewColumn)  8 'Column 1' 303 8 #left 1552 8 ##(Smalltalk.SortedCollection)  0 0 1040 0 3 0 0 8 #report 976 0 133217 0 0 754 202 208 98 3 818 848 98 2 882 1 1 882 311 831 1040 818 8 #contextMenu: 98 1 1264 1040 818 8 #text: 98 1 8 'Column 1' 1040 930 8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 0 0 0 0 0 0 0 0 155 0 0 0 159 1 0 0] 98 0 992 0 27 8 'FrameList' 608 8 'StackFrame' 0 754 202 208 98 1 818 848 98 2 882 11 11 882 941 831 416 930 8 #[44 0 0 0 0 0 0 0 0 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 5 0 0 0 5 0 0 0 219 1 0 0 164 1 0 0] 98 3 1040 410 8 ##(Smalltalk.Splitter)  98 12 0 416 98 2 8 1140850688 1 2208 0 498 528 0 519 0 0 0 2208 754 202 208 98 1 818 848 98 2 882 311 1 882 7 831 2208 930 8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 155 0 0 0 0 0 0 0 158 0 0 0 159 1 0 0] 98 0 992 0 27 608 992 0 27 )! !
!GhoulStackTracePresenter class categoriesFor: #defaultModel!constants!public! !
!GhoulStackTracePresenter class categoriesFor: #resource_Default_view!public!resources-views! !

GhoulShell guid: (GUID fromString: '{DB301CC6-1CF3-4526-944C-EB8E133E084F}')!
GhoulShell comment: 'Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org

Tool that provides a debugger-like view of Dolphin crash dump files or the .ERRORS files produced by unhandled errors in deployed executables.

It uses the definitions of the methods in the current image to interpret the stack trace in the .ERRORS file, and so links the stack frames to the source for their methods, and the correct variable names.  It should handle the case where the infomation is missing OK (but obviously with reduced functionality).  It will become confused if the source has changed between the deployed version and what''s in the current image.

It will install itself as an ''extra tool'' and can be started from the Dolhpin system folder in the normal way.

Please do not try to open files that are not either Dolphin crash dumps, or produced by SessioinManager>>logError (I.e. using VMLibrary>>dump:...).  In particular, the "normal" .errors file that a development image writes to (in the same directory as the change log, etc) is /not/ suitable for opening with this tool.'!
!GhoulShell categoriesForClass!Unclassified! !
!GhoulShell methodsFor!

baseForDecodingIpNumbers
	"ugh!!  Answer the base that we will use for decoding IP numbers.
	This stuff only exists because Dolphin is nearly, but not quite, reliable
	about how it encodes IP numbers in stack frames"

	^ self stackTracePresenter model ipStringBase.!

baseForDecodingIpNumbers: anInteger
	"ugh!!  Set the base that we will use for decoding IP numbers.
	This stuff only exists because Dolphin is nearly, but not quite, reliable
	about how it encodes IP numbers in stack frames"

	| presenter |

	presenter := self stackTracePresenter.

	presenter model ipStringBase: anInteger.
	presenter onFrameSelected.		#CUtodo.  "ugly, ugly, ugly encapsulation violation..."
!

buildRecentFilesMenu: aMenu
	"private -- this is invoked when the dynamic recent files menu is about to be displayed;
	update it appropriately"

	| recent command item text |

	aMenu clear.		"we build it fresh each time"

	recent := self class recentFiles asSortedCollection.
	recent keysAndValuesDo:
		[:i :each |
		text := '&' , i displayString , ' - ' , each displayString.
		command := Message selector: #fileOpen: argument: each.
		item := CommandMenuItem command: command description: text.
		aMenu addItem: item].

	recent notEmpty ifTrue: [aMenu addSeparator].

	text := '&Clipboard'.
	command := Message selector: #fileOpenClipboard.
	item := CommandMenuItem command: command description: text.
	aMenu addItem: item.
!

canDecodeIpNumbers
	"private -- answer whether we may perform the #decodeIpNumbersAs{Hex/Decimal} commands"

	^ self stackTracePresenter model stackFrames notEmpty.
!

canFileRefresh
	"private -- answer whether we may perform the #fileRefresh command"

	^ self model canRefresh.!

canOpenFilename: aFilename
	"private -- answer whether we know how to open the named file"

	"currently we don't attempt to restrict the names of files we can open"
	^ true.!

checkForExternalModifications
	"private -- check for changes to the external representation of our model"

	| filename msg |

	self model externalFileHasChanged ifFalse: [^ self].
	filename := self model filename.
	self model resetInternalTimestamp.

	msg := (String writeStream)
			nextPutAll: 'The file "';
			display: filename;
			nextPutAll: '" had been changed on disk.';
			cr;
			nextPutAll: 'Do you want to load this new version ?';
			contents.
	(self confirm: msg) ifTrue: [self fileRefresh].!

connectDragAndDrop
	"private -- listen for drag and drop from Windows"

	| sdds |

	sdds := Smalltalk at: #ShellDragDropSession ifAbsent: [^ self]
.
	self droppableSubpresenters do:
		[:each |
		sdds registerDropTarget: each view.
		each view isDropTarget: true.
		each
			when: #dragOver: send: #onDragOverShell: to: self;
			when: #drop: send: #onDropOverShell: to: self].
!

couldNotFindModelIn: aFilename
	"private -- popup an error message to say that we failed to find a model in the named file,
	will also give the option to remove it from the recent file list"

	| warn cap ok |

	warn := 'Could not find any stack traces in ' , aFilename.
	cap := self class toolName.

	(self class recentFiles includes: aFilename) ifFalse:
		[^ MessageBox warning: warn caption: cap].

	warn := warn , '

Remove it from the "Recent files" list ?'.

	"we use a ok/cancel in order that <ESC> will still kill the popup"
	ok := (MessageBox new)
		caption: cap;
		text: warn;
		warning;
		okCancel;
		open.

	ok = #ok ifTrue: [self class removeFromRecentFiles: aFilename].
		
!

couldNotOpen: aFilename because: anError
	"private -- popup an error message to say that we failed to open the named file,
	will also give the option to remove it from the recent file list"

	| warn cap ok |

	warn := 'Could not open or read ' , aFilename , ' (' , anError displayString , ')'.
	cap := self class toolName.

	(self class recentFiles includes: aFilename) ifFalse:
		[^ MessageBox warning: warn caption: cap].

	warn := warn , '

Remove it from the "Recent files" list ?'.

	"we use a ok/cancel in order that <ESC> will still kill the popup"
	ok := (MessageBox new)
		caption: cap;
		text: warn;
		warning;
		okCancel;
		open.

	ok = #ok ifTrue: [self class removeFromRecentFiles: aFilename].
		
!

createComponents
	"private -- create presenters in order that they may be bound into MVP triads"

	self
		add: (GhoulStackTracePresenter new) name: 'StackTrace';
		add: (ListPresenter new) name: 'TraceList';
		yourself.

	^ super createComponents.
!

createSchematicWiring
	"private -- arrange triggering between our components"

	self traceListPresenter when: #selectionChanged send: #onTraceSelected to: self.

	self connectDragAndDrop.

	^ super createSchematicWiring.
!

decodeIpNumbersAs: anInteger
	"command -- decode ip numbers as being in the given base"

	self canDecodeIpNumbers ifFalse: [^ self].

	self baseForDecodingIpNumbers: anInteger.!

decodeIpNumbersAsDecimal
	"command -- decode ip numbers as decimal"

	self decodeIpNumbersAs: 10.!

decodeIpNumbersAsHex
	"command -- decode ip numbers as hex"

	self decodeIpNumbersAs: 16.!

disconnectDragAndDrop
	"private -- decline drag and drop notifications from Windows"

	| sdds |

	sdds := Smalltalk at: #ShellDragDropSession ifAbsent: [^ self]
.
	"the error trap is because ShellDragDropSession may have closed down before us as the image is shut down"
	[self droppableSubpresenters do: [:each | sdds revokeDropTarget: each view]]
		on: Error
		do: [:err | ].
!

droppableSubpresenters
	"private -- answer a collection of subpresenters over which we will
	allow dropping of crashdump files from Windows"

	"why not all the components ?"
	^ Array
		with: self
		with: self stackTracePresenter frameListPresenter.!

fileClose
	"command -- close the current window. Since we have no interesting state we don't
	bother with Are-You-Really-Sure? prompts"

	self view close.!

fileNewWindow
	"command -- open a new window on the same model"

	self class showOn: self model.!

fileOpen
	"command -- prompt for a post-mortem file to open and open it in this window"

	| initialDirectory filename zipfile |

	self model isVirtual ifFalse:
		[initialDirectory := File splitPathFrom: self model filename].

	filename := (FileOpenDialog new)
				fileTypes: GhoulModel fileTypes;
				caption: 'Open post-mortem file...';
				initialDirectory: initialDirectory;
				showModal.

	filename isNil ifFalse: [self fileOpen: filename].
!

fileOpen: aFilename
	"command -- open a file in this window if possible"

	| newModel |

	[newModel := GhoulModel fromFile: aFilename]
		on: Error
		do: [:err | ^ self couldNotOpen: aFilename because: err].

	newModel isNil ifTrue: [^ self].
	newModel stackTraces isEmpty ifTrue: [^ self couldNotFindModelIn: aFilename].

	self model: newModel.
!

fileOpenClipboard
	"command -- attempt to display the current contents of the clipboard"

	| newModel |

	[newModel := VirtualGhoulModel fromClipboard]
		on: Error
		do: [:err | ^ MessageBox
				warning: ('Could not read a stack trace from the clipboard (' , err displayString , ')')
				caption: self class toolName].

	newModel isNil ifTrue: [^ self].
	newModel stackTraces isEmpty ifTrue:
		[^ MessageBox
			warning: ('Could not find any stack traces in the clipboard''s contents')
			caption: self class toolName].

	self model: newModel.
!

fileRefresh
	"command -- file refresh"

	self canFileRefresh ifFalse: [^ self].

	[self model refresh]
		on: Error
		do: [:err | MessageBox
				warning: ('Could not refresh from ' , self model displayString , ' (' , err displayString , ')')
				caption: self class toolName].

	self model: self model.!

filesFromDDSession: aDragDropSession
	"private -- extract the openable filenames from aDragDropSession"

	| objects object files |

	objects := aDragDropSession dragObjects select: [:each | each isFormatAvailable: #Filenames].
	files := OrderedCollection new.
	objects do: [:each | files addAll: (each format: #Filenames)].
	files := files select: [:each | self canOpenFilename: each].

	^ files.!

help
	"command -- display the help for this tool"

	| locator file |

	locator := PackageRelativeFileLocator package: self class owningPackage.
	file := locator localFileSpecFor: ('Docs\Ghoul.html').
	[ShellLibrary default shellOpen: file]
		on: Error
		do: [:err | MessageBox
			notify: ('Sorry.  No help is available for Ghoul')
			caption: 'Ghoul Help'].

!

inspectOptions
	"open a PAI on the system options."

	(PublishedAspectInspector shellOn: self class) topShell
		caption: (self class name , ' Options').
	!

model: aGhoulModel
	"set the model that we will display"

	| traces |

	super model: aGhoulModel.

	traces := aGhoulModel stackTraces.
	self traceListPresenter list: traces.
	traces isEmpty ifFalse: [self traceListPresenter selectionByIndex: traces size].

	self captionExtension: aGhoulModel displayString.

	aGhoulModel isVirtual ifFalse: [self class addToRecentFiles: aGhoulModel filename].!

onAboutToDisplayMenu: aMenu
	"private -- this is invoked when aMenu is about to be popped-up;
	update it appropriately"

	| name |

	name := aMenu name.
	name == #dynamicRecentFilesMenu ifTrue: [^ self buildRecentFilesMenu: aMenu].

!

onDragOverShell: aDragDropSession
	"private -- handler for when Windows is dragging something over our shell"

	| files |

	files := self filesFromDDSession: aDragDropSession.

	files size = 1 ifTrue: [aDragDropSession operation: #move].!

onDropOverShell: aDragDropSession
	"private -- handler for when Windows is droping something over our shell"

	| files |

	files := self filesFromDDSession: aDragDropSession.

	files notEmpty ifTrue: [Cursor wait showWhile: [self fileOpen: files first]].

	"we never do anything that should cause the source to 'cut' what was dropped"
	aDragDropSession operation: nil.
!

onTraceSelected
	"private -- a stack trace has been selected, update the trace presenter
	accordingly"

	self stackTracePresenter model: self selectedTraceOrDummy.
!

onViewActivated: anEvent
	"handler for activation, overridden to check for modifications to the saved file"

	[self checkForExternalModifications] postToInputQueue.

	^ super onViewActivated: anEvent.!

onViewClosed
	"private -- our view has been closed, clean up"

	super onViewClosed.

	self disconnectDragAndDrop.!

onViewOpened
	"private -- last-minute settup"

	super onViewOpened.

	"Grrr...."
	self model: self model.!

queryCommand: aCommandQuery
	"set the enabledness of the command represented by aCommandQuery"

	| cmd enabled checked |

	super queryCommand: aCommandQuery.

	cmd := aCommandQuery commandSymbol.
	enabled := aCommandQuery isEnabled.
	checked := aCommandQuery isChecked.

	cmd == #fileRefresh ifTrue: [enabled := self canFileRefresh].
	cmd = #dynamicRecentFilesMenu ifTrue: [enabled := true].
	cmd == #decodeIpNumbersAsDecimal ifTrue:
			[enabled := self canDecodeIpNumbers.
			checked := self baseForDecodingIpNumbers = 10].
	cmd == #decodeIpNumbersAsHex ifTrue:
			[enabled := self canDecodeIpNumbers.
			checked := self baseForDecodingIpNumbers = 16].

	aCommandQuery 
		isEnabled: enabled;
		isChecked: checked.
!

selectedTraceOrDummy
	"private -- answer the currently selected stack trace, if there is one,
	or a dummy, if not"

	^ self traceListPresenter selectionIfNone: [GhoulStackTrace dummy].!

stackTracePresenter
	"private -- answer the presenter named 'StackTrace'"

	^ self presenterNamed: 'StackTrace'.
!

traceListPresenter
	"private -- answer the presenter named 'TraceList'"

	^ self presenterNamed: 'TraceList'.
! !
!GhoulShell categoriesFor: #baseForDecodingIpNumbers!accessing!public! !
!GhoulShell categoriesFor: #baseForDecodingIpNumbers:!accessing!public! !
!GhoulShell categoriesFor: #buildRecentFilesMenu:!menus!private! !
!GhoulShell categoriesFor: #canDecodeIpNumbers!commands!private! !
!GhoulShell categoriesFor: #canFileRefresh!commands!private! !
!GhoulShell categoriesFor: #canOpenFilename:!helpers!private! !
!GhoulShell categoriesFor: #checkForExternalModifications!helpers!private! !
!GhoulShell categoriesFor: #connectDragAndDrop!drag and drop!event handling!initializing!private! !
!GhoulShell categoriesFor: #couldNotFindModelIn:!helpers!private! !
!GhoulShell categoriesFor: #couldNotOpen:because:!helpers!private! !
!GhoulShell categoriesFor: #createComponents!initializing!private!subpresenters! !
!GhoulShell categoriesFor: #createSchematicWiring!event handling!initializing!private!subpresenters! !
!GhoulShell categoriesFor: #decodeIpNumbersAs:!commands!public! !
!GhoulShell categoriesFor: #decodeIpNumbersAsDecimal!commands!public! !
!GhoulShell categoriesFor: #decodeIpNumbersAsHex!commands!public! !
!GhoulShell categoriesFor: #disconnectDragAndDrop!drag and drop!event handling!initializing!private! !
!GhoulShell categoriesFor: #droppableSubpresenters!drag and drop!private!subpresenters! !
!GhoulShell categoriesFor: #fileClose!commands!public! !
!GhoulShell categoriesFor: #fileNewWindow!commands!public! !
!GhoulShell categoriesFor: #fileOpen!commands!public! !
!GhoulShell categoriesFor: #fileOpen:!commands!public! !
!GhoulShell categoriesFor: #fileOpenClipboard!commands!public! !
!GhoulShell categoriesFor: #fileRefresh!commands!public! !
!GhoulShell categoriesFor: #filesFromDDSession:!drag and drop!private! !
!GhoulShell categoriesFor: #help!commands!public! !
!GhoulShell categoriesFor: #inspectOptions!commands!public! !
!GhoulShell categoriesFor: #model:!initializing!models!public! !
!GhoulShell categoriesFor: #onAboutToDisplayMenu:!event handling!menus!private! !
!GhoulShell categoriesFor: #onDragOverShell:!drag and drop!event handling!private! !
!GhoulShell categoriesFor: #onDropOverShell:!drag and drop!event handling!private! !
!GhoulShell categoriesFor: #onTraceSelected!event handling!private! !
!GhoulShell categoriesFor: #onViewActivated:!event handling!private! !
!GhoulShell categoriesFor: #onViewClosed!event handling!private! !
!GhoulShell categoriesFor: #onViewOpened!event handling!initializing!private! !
!GhoulShell categoriesFor: #queryCommand:!commands!menus!public! !
!GhoulShell categoriesFor: #selectedTraceOrDummy!accessing!private! !
!GhoulShell categoriesFor: #stackTracePresenter!private!subpresenters! !
!GhoulShell categoriesFor: #traceListPresenter!private!subpresenters! !

!GhoulShell class methodsFor!

about
	"answer a string very briefly describing ourself"

	^ 'Dolphin Post-Mortem File Viewer.  Version 3.
Copyright  Chris Uppal, 2004, 2005.
chris.uppal@metagnostic.org'.!

addToRecentFiles: aFilename
	"append the filename to our list of recently opened files"

	self removeFromRecentFiles: aFilename.

	recentFiles addFirst: aFilename.
	[recentFiles size > 10] whileTrue: [recentFiles removeLast].!

bugs
	"answer a String describing the less than outstanding work"

	^ '
Can''t parse time stamps of the form "17:53:33 PM" (Dolphin''s time parser barfs).
I can''t make the drag-and-drop stuff work reliably over image restart.
'.!

clearRecentFiles
	"clear our list of the names of recently opened files.

		self clearRecentFiles.
	"

	recentFiles := OrderedCollection new.!

defaultFadeFactor
	"answer the fade factor to use by default"

	^ 2.	"slightly faded"!

defaultIconName
	"answers the name Icon that can be used to represent this class"

	^ 'Ghoul.ico'.!

defaultModel

	^ VirtualGhoulModel new.!

dumpFileDateFormat
	"answer the date format we expect to find in duimp files.  This is normally
	nil (in which case the format is read from the locale) but can be set explicitly.
	NB: under Win2K, it seems that Dolphin always ignores the locale and writes
	the timestamp in US form"

	^ (self registryEntry at: #dumpFileDateFormat ifAbsent: [^ nil]) value.
!

dumpFileDateFormat: aStringOrNil
	"answer the date format we expect to find in duimp files.  This is normally
	nil (in which case the format is read from the locale) but can be set explicitly.
	NB: under Win2K, it seems that Dolphin always ignores the locale and writes
	the timestamp in US form

		OSVERSIONINFO current isWinXP ifFalse: [self dumpFileDateFormat: 'MM/dd/yyyy']
	"

	(aStringOrNil isNil or: [aStringOrNil isEmpty])
		ifTrue: [self removeRegistryAt: #dumpFileDateFormat]
		ifFalse: [self registryAt: #dumpFileDateFormat put: aStringOrNil].!

help
	"answer a string describing ourselves"

	^ '
Simple tool for looking at the contents of post-mortem
files such as those produced by deployed Dolphin applications.

	-- chris
'.
!

initialize
	"private -- class-side initialisation.

		self initialize.
	"

	self clearRecentFiles.
	self reuseIfOpen: false.
	self registerAsTool.
!

openFile: aFilename
	"show a new instance on the contents of the named file"

	^ (self show)
		fileOpen: aFilename;
		yourself.!

publishedAspects
	"answer our aspects as a Set"

	^ super publishedAspects
		add: (Aspect integer: #sourcePaneFadeFactor);
		add: (Aspect string: #dumpFileDateFormat);
		yourself.
!

recentFiles
	"answer a list of the names of recently opened files"

	^ recentFiles.!

removeFromRecentFiles: aFilename
	"ensure the filename is not on our list of recently opened files"

	recentFiles := recentFiles reject: [:each | each sameAs: aFilename].
!

resource_Default_view
	"Answer the literal data from which the 'Default view' resource can be reconstituted.
	DO NOT EDIT OR RECATEGORIZE THIS METHOD.

	If you wish to modify this resource evaluate:
	ViewComposer openOn: (ResourceIdentifier class: self selector: #resource_Default_view)
	"

	^#(#'!!STL' 3 788558 10 ##(Smalltalk.STBViewProxy)  8 ##(Smalltalk.ShellView)  98 27 0 0 98 2 27131905 131073 416 0 524550 ##(Smalltalk.ColorRef)  8 4278190080 0 517 0 0 0 416 788230 ##(Smalltalk.BorderLayout)  1 1 410 8 ##(Smalltalk.ComboBox)  98 17 0 416 98 2 8 1144063491 1025 560 590662 2 ##(Smalltalk.ListModel)  202 208 98 0 0 1114638 ##(Smalltalk.STBSingletonProxy)  8 ##(Smalltalk.SearchPolicy)  8 #identity 482 512 0 5 0 0 0 560 0 8 4294902129 8 ##(Smalltalk.BasicListAbstract)  688 401 983302 ##(Smalltalk.MessageSequence)  202 208 98 1 721670 ##(Smalltalk.MessageSend)  8 #createAt:extent: 98 2 328198 ##(Smalltalk.Point)  1 1 946 1415 43 560 983302 ##(Smalltalk.WINDOWPLACEMENT)  8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 0 0 0 0 0 0 0 0 195 2 0 0 21 0 0 0] 98 0 946 193 193 0 27 0 0 0 410 8 ##(Smalltalk.ReferenceView)  98 14 0 416 98 2 8 1140850688 131073 1072 0 0 0 5 0 0 0 1072 1180166 ##(Smalltalk.ResourceIdentifier)  8 ##(Smalltalk.GhoulStackTracePresenter)  8 #resource_Default_view 0 818 202 208 98 1 882 912 98 2 946 1 43 946 1415 1439 1072 994 8 #[44 0 0 0 0 0 0 0 1 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 0 0 0 0 21 0 0 0 195 2 0 0 228 2 0 0] 688 1056 0 27 234 256 98 4 560 8 'TraceList' 1072 8 'StackTrace' 0 461638 4 ##(Smalltalk.MenuBar)  0 16 98 5 265030 4 ##(Smalltalk.Menu)  0 16 98 7 984134 2 ##(Smalltalk.CommandMenuItem)  1 1180998 4 ##(Smalltalk.CommandDescription)  8 #fileNewWindow 8 '&New window' 9373 1 0 0 0 1522 1 1554 8 #fileOpen 8 '&Open...' 9375 1 0 0 0 1522 1 1554 8 #fileRefresh 8 'Refres&h' 1257 1 0 0 0 983366 1 ##(Smalltalk.DividerMenuItem)  4097 1474 0 16 98 0 8 '&Recent' 8 #dynamicRecentFilesMenu 1 0 0 15833 0 0 1746 4097 1522 1 1554 8 #fileClose 8 'Close' 17639 1 0 0 0 8 '&File' 0 1 0 0 15837 0 0 1474 0 16 98 3 1522 1 1554 8 #browseIt 8 '&Browse It' 9349 1 0 0 0 1522 1 1554 8 #browseDefinitions 8 'Browse &definitions' 1271 1 0 0 0 1522 1 1554 8 #browseReferences 8 'Browse &references' 5367 1 0 0 0 8 '&Browse' 0 1 0 0 15845 0 0 1474 0 16 98 1 1474 0 16 98 2 1522 1025 1554 8 #decodeIpNumbersAsHex 8 '&Hex' 13489 1 0 0 0 1522 1025 1554 8 #decodeIpNumbersAsDecimal 8 '&Decimal' 13449 1 0 0 0 8 'Decode &IP numbers as' 0 1 0 0 15851 0 0 8 '&View' 0 1 0 0 15853 0 0 1474 0 16 98 1 1474 0 16 98 6 1522 1 1554 8 #toggleTopMost 8 'Always on &Top' 8361 1 0 0 0 1746 4097 1522 1 1554 8 #rememberWindowSize 8 '&Remember this size' 1 1 0 0 0 1522 1 1554 8 #forgetWindowSize 8 '&Forget size' 1 1 0 0 0 1746 4097 1522 1 1554 8 #inspectOptions 8 '&Inspect options' 1 1 0 0 0 8 '&Options' 0 1 0 0 15863 0 0 8 '&Tools' 0 1 0 0 15865 0 0 1474 0 16 98 6 1522 1 1554 8 #help 8 '&Help on this tool' 225 1 0 0 0 1746 4097 1522 1 1554 8 #helpAbout 8 '&About this tool' 1 1 0 0 0 1746 4097 1522 1 1554 8 #bugs 8 '&Bugs' 1 1 0 0 0 1522 1 1554 8 #todo 8 '&Todo' 1 1 0 0 0 8 '&Help' 0 1 0 0 15875 0 0 8 '' 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 818 202 208 98 3 882 912 98 2 946 177 233 946 2881 1701 416 882 8 #text: 98 1 8 'Ghoul' 416 882 8 #updateMenuBar 688 416 994 8 #[44 0 0 0 0 0 0 0 0 0 0 0 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 88 0 0 0 116 0 0 0 248 5 0 0 198 3 0 0] 98 2 560 1072 1056 0 27 )!

sourcePaneFadeFactor
	"answer the amount (default 2) by which we fade the background color in the
	source pane"

	^ (self registryEntry at: #sourcePaneFadeFactor ifAbsent: [^ self defaultFadeFactor]) value.
!

sourcePaneFadeFactor: anIntegerOrNil
	"set the amount (default 2) by which we fade the background color in the
	source pane.

	This expression turns off fading entirely:

		self sourcePaneFadeFactor: 1
	"

	(anIntegerOrNil isNil or: [anIntegerOrNil = self defaultFadeFactor])
		ifTrue: [self removeRegistryAt: #sourcePaneFadeFactor]
		ifFalse: [self registryAt: #sourcePaneFadeFactor put: anIntegerOrNil].

	"hacky..."
	GhoulStackFramePresenter onSettingsChanged.!

todo
	"answer a String describing the outstanding work"

	^ '
Should keep the recent files list in the registry.
Make browse* commands do something sensible in the stack frame presenter.
Would doing our own time/date parsing would be less fragile ?
Or maybe remove the timestamp parsing altogether ?
'.!

uninitialize
	"private -- class tear-down.

		self uninitialize.
	"

	recentFiles := nil.
	self unRegisterAsTool.! !
!GhoulShell class categoriesFor: #about!documentation!public! !
!GhoulShell class categoriesFor: #addToRecentFiles:!public!recent files! !
!GhoulShell class categoriesFor: #bugs!documentation!public! !
!GhoulShell class categoriesFor: #clearRecentFiles!public!recent files! !
!GhoulShell class categoriesFor: #defaultFadeFactor!constants!public! !
!GhoulShell class categoriesFor: #defaultIconName!constants!public! !
!GhoulShell class categoriesFor: #defaultModel!constants!public! !
!GhoulShell class categoriesFor: #dumpFileDateFormat!accessing!public!registry! !
!GhoulShell class categoriesFor: #dumpFileDateFormat:!accessing!public!registry! !
!GhoulShell class categoriesFor: #help!documentation!public! !
!GhoulShell class categoriesFor: #initialize!development!initializing!private! !
!GhoulShell class categoriesFor: #openFile:!instance creation!public! !
!GhoulShell class categoriesFor: #publishedAspects!constants!must strip!public! !
!GhoulShell class categoriesFor: #recentFiles!public!recent files! !
!GhoulShell class categoriesFor: #removeFromRecentFiles:!public!recent files! !
!GhoulShell class categoriesFor: #resource_Default_view!public!resources-views! !
!GhoulShell class categoriesFor: #sourcePaneFadeFactor!accessing!public!registry! !
!GhoulShell class categoriesFor: #sourcePaneFadeFactor:!accessing!public!registry! !
!GhoulShell class categoriesFor: #todo!documentation!public! !
!GhoulShell class categoriesFor: #uninitialize!development!initializing!private! !

"Binary Globals"!

